import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';

export default class WindowAutoHideExtension {
    constructor() {
        this._focusWindowId = null;
        this._windowCreatedId = null;
    }

    enable() {
        // Connect to focus-window signal to handle window switching
        this._focusWindowId = global.display.connect(
            'notify::focus-window',
            this._onFocusChanged.bind(this)
        );

        // Also handle newly created windows
        this._windowCreatedId = global.display.connect(
            'window-created',
            this._onWindowCreated.bind(this)
        );

        // Hide all non-active windows on enable
        this._hideNonActiveWindows();
    }

    disable() {
        // Disconnect signals
        if (this._focusWindowId) {
            global.display.disconnect(this._focusWindowId);
            this._focusWindowId = null;
        }
        if (this._windowCreatedId) {
            global.display.disconnect(this._windowCreatedId);
            this._windowCreatedId = null;
        }

        // Show all windows when extension is disabled
        this._showAllWindows();
    }

    _onFocusChanged(display) {
        this._hideNonActiveWindows();
    }

    _onWindowCreated(display, window) {
        // Resize and center the window
        this._resizeAndCenterWindow(window);
        
        // When a new window is created, update visibility
        window.connect('shown', () => {
            this._hideNonActiveWindows();
        });
    }

    _hideNonActiveWindows() {
        let activeWindow = global.display.focus_window;
        let windows = global.get_window_actors();

        windows.forEach(windowActor => {
            let window = windowActor.meta_window;

            // Skip window types that should always be visible
            let windowType = window.get_window_type();
            if (windowType === Meta.WindowType.DESKTOP ||
                windowType === Meta.WindowType.DOCK ||
                windowType === Meta.WindowType.POPUP_MENU ||
                windowType === Meta.WindowType.DROPDOWN_MENU ||
                windowType === Meta.WindowType.TOOLTIP ||
                windowType === Meta.WindowType.NOTIFICATION ||
                windowType === Meta.WindowType.COMBO ||
                windowType === Meta.WindowType.DND ||
                windowType === Meta.WindowType.OVERRIDE_OTHER) {
                windowActor.show();
                return;
            }

            // Hide non-active windows, show the active one
            if (window === activeWindow) {
                windowActor.show();
            } else {
                windowActor.hide();
            }
        });
    }

    _resizeAndCenterWindow(window) {
        // Skip window types that shouldn't be resized
        let windowType = window.get_window_type();
        if (windowType !== Meta.WindowType.NORMAL) {
            return;
        }

        // Get the monitor where the window is located
        let monitorIndex = window.get_monitor();
        let monitor = Main.layoutManager.monitors[monitorIndex];

        if (!monitor) {
            return;
        }

        // Calculate new size: 50% width, 66.67% height
        let newWidth = Math.floor(monitor.width * 0.5);
        let newHeight = Math.floor(monitor.height * (2 / 3));

        // Calculate position to center the window
        let newX = monitor.x + Math.floor((monitor.width - newWidth) / 2);
        let newY = monitor.y + Math.floor((monitor.height - newHeight) / 2);

        // Resize and move the window
        window.move_resize_frame(false, newX, newY, newWidth, newHeight);
    }

    _showAllWindows() {
        let windows = global.get_window_actors();

        windows.forEach(windowActor => {
            windowActor.show();
        });
    }
}
